/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_sqldbms.h"
#include "udm_utils.h"
#include "udm_url.h"
#include "udm_sdp.h"
#include "udm_vars.h"
#include "udm_mutex.h"
#include "udm_searchtool.h"
#include "udm_result.h"
#include "udm_log.h"

#include "udm_proto.h"
#include "udm_host.h"
#include "udm_crc32.h"
#include "udm_doc.h"

#define UDM_THREADINFO(A,s,m)	if(A->Conf->ThreadInfo)A->Conf->ThreadInfo(A,s,m)


void *UdmDBInit(void *vdb){
	UDM_DB *db=vdb;
	size_t	nbytes=sizeof(UDM_DB);
	
	if(!db){
		db=(UDM_DB*)malloc(nbytes);
		bzero(db,nbytes);
		db->freeme=1;
	}else{
		bzero(db,nbytes);
	}
	db->numtables=32;
	
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
	db->hDbc=SQL_NULL_HDBC;
	db->hEnv=SQL_NULL_HENV;
	db->hstmt=SQL_NULL_HSTMT;
#endif
#if (HAVE_IBASE)
	db->DBH=NULL;
#endif
#if (HAVE_ORACLE8)
	db->par = UdmXmalloc(sizeof(struct param_struct));
#endif
	return db;
}


void UdmDBFree(void *vdb){
	UDM_DB	*db=vdb;
	
	UDM_FREE(db->DBADDR);
	UDM_FREE(db->DBName);
	UDM_FREE(db->DBUser);
	UDM_FREE(db->DBPass);
	UDM_FREE(db->DBSock);
	UDM_FREE(db->where);
	UDM_FREE(db->TargetsQuery);
	
	if(db->DBMode==UDM_DBMODE_CACHE)UdmCloseCache(db);
	
	if(!db->connected)goto ret;
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES){
		size_t i;
		if(db->dict)fclose(db->dict);
		if(db->url)fclose(db->url);
		for(i=0;i<MAXMULTI;i++)
			if(db->crcdict[i]>=0)
				close(db->crcdict[i]);
	}
#endif
	
#if HAVE_MYSQL
	if(db->DBDriver==UDM_DB_MYSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_PGSQL
	if(db->DBDriver==UDM_DB_PGSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_MSQL
	if(db->DBDriver==UDM_DB_MSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_IBASE
	if(db->DBDriver==UDM_DB_IBASE){
		UdmSQLClose(db);
		goto ret;
	}
#endif

#if HAVE_ORACLE8
	if(db->DBDriver==UDM_DB_ORACLE8){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_ORACLE7
	if(db->DBDriver==UDM_DB_ORACLE7){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if HAVE_CTLIB
	if(db->DBDriver==UDM_DB_MSSQL){
		UdmSQLClose(db);
		goto ret;
	}
#endif
	
#if (HAVE_IODBC || HAVE_UNIXODBC || HAVE_SOLID || HAVE_VIRT || HAVE_EASYSOFT || HAVE_SAPDB || HAVE_DB2)
	UdmSQLClose(db);
	goto ret;
#endif
ret:
	if(db->freeme)free(vdb);
	return;
}



__INDLIB__ UDM_UINT8_URLID *UdmLimit8(UDM_AGENT *Agent,const char *field, int *count, int type,void *vdb){
	UDM_UINT8_URLID	*res=NULL;
	UDM_DB		*db=vdb;
	
	*count=0;
#ifdef HAVE_SQL
	res=UdmLimit8SQL(Agent,field,count,type,db);
#endif
	strcpy(Agent->Conf->errstr,db->errstr);
	Agent->Conf->errcode=db->errcode;
	return res;
}

__INDLIB__ UDM_UINT4_URLID *UdmLimit4(UDM_AGENT *Agent,const char *field, int *count, int type, void *vdb){
	UDM_UINT4_URLID	*res=NULL;
	UDM_DB		*db=vdb;
	
	*count=0;
#ifdef HAVE_SQL
	res=UdmLimit4SQL(Agent,field,count,type,db);
#endif
	strcpy(Agent->Conf->errstr,db->errstr);
	Agent->Conf->errcode=db->errcode;
	return res;
}


__INDLIB__ int UdmClearDatabase(UDM_AGENT *A){
	int	res=UDM_ERROR;
	UDM_DB	*db=A->Conf->db;

#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmClearDBFiles(A,db);
#endif
#ifdef HAVE_SQL
	if(((UDM_DB*)(A->Conf->db))->DBDriver!=UDM_DB_FILES)
		res=UdmClearDBSQL(A,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}



static int DocUpdate(UDM_AGENT * Indexer, UDM_DOCUMENT *Doc, void *db){
	UDM_HOST_ADDR	*Host=NULL;
	char		site_id_str[UDM_URLSIZE]="";
	int		result=UDM_OK;
	const char	*c;
	int		status=UdmVarListFindInt(&Doc->Sections,"Status",0);
	int		origin_id=0;
	int		url_id=UdmVarListFindInt(&Doc->Sections,"ID",0);
	time_t		next_index_time;
	char		dbuf[64];
	int		use_crosswords=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"CrossWords","no"),"yes");
	int		use_newsext=!strcasecmp(UdmVarListFindStr(&Indexer->Conf->Vars,"NewsExtensions","no"),"yes");
	
	/* First of all check that URL must be delated */
	
	if(Doc->method==UDM_METHOD_DISALLOW){
		UdmLog(Indexer,UDM_LOG_ERROR,"Deleting");
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_DELETE,db);
		return result;
	}
	
	next_index_time=time(NULL)+Doc->Spider.period;
	UdmTime_t2HttpStr(next_index_time,dbuf);
	UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
	
	switch(status){
	
	case 0: /* No HTTP code */
		if((Host=UdmHostFind(&Indexer->Conf->Hosts,Doc->CurURL.hostname))){
			Host->net_errors++;
			Host=NULL;
		}
		UdmLog(Indexer,UDM_LOG_ERROR,"No HTTP response status");
		next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
		UdmTime_t2HttpStr(next_index_time,dbuf);
		UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
	
	case UDM_HTTP_STATUS_OK:				/* 200 */
	case UDM_HTTP_STATUS_PARTIAL_OK:			/* 206 */
	        Host = UdmHostFind(&Indexer->Conf->Hosts, Doc->CurURL.hostname);
		if(!UdmVarListFind(&Doc->Sections,"Content-Type")){
			UdmLog(Indexer,UDM_LOG_ERROR,"No Content-type header");
			next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
			UdmTime_t2HttpStr(next_index_time,dbuf);
			UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
			UdmVarListReplaceInt(&Doc->Sections,"Status",UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR);
			
			if(Host){
				Host->net_errors++;
				Host=NULL;
			}
			result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
			return result;
		} else {
		        if (Host) {
				Host->net_errors = 1;
				Host = NULL;
			}
		}
		break;
	
	case UDM_HTTP_STATUS_MULTIPLE_CHOICES:			/* 300 */
	case UDM_HTTP_STATUS_MOVED_PARMANENTLY:			/* 301 */
	case UDM_HTTP_STATUS_MOVED_TEMPORARILY:			/* 302 */
	case UDM_HTTP_STATUS_SEE_OTHER:				/* 303 */
	case UDM_HTTP_STATUS_NOT_MODIFIED:			/* 304 */
		/* FIXME: check that status is changed and remove words if necessary */
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
		break;
	
	case UDM_HTTP_STATUS_USE_PROXY:				/* 305 */
	case UDM_HTTP_STATUS_BAD_REQUEST:			/* 400 */
	case UDM_HTTP_STATUS_UNAUTHORIZED:			/* 401 */
	case UDM_HTTP_STATUS_PAYMENT_REQUIRED:			/* 402 */
	case UDM_HTTP_STATUS_FORBIDDEN:				/* 403 */
	case UDM_HTTP_STATUS_NOT_FOUND:				/* 404 */
	case UDM_HTTP_STATUS_METHOD_NOT_ALLOWED:		/* 405 */
	case UDM_HTTP_STATUS_NOT_ACCEPTABLE:			/* 406 */
	case UDM_HTTP_STATUS_PROXY_AUTHORIZATION_REQUIRED:	/* 407 */
	case UDM_HTTP_STATUS_REQUEST_TIMEOUT:			/* 408 */
	case UDM_HTTP_STATUS_CONFLICT:				/* 409 */
	case UDM_HTTP_STATUS_GONE:				/* 410 */
	case UDM_HTTP_STATUS_LENGTH_REQUIRED:			/* 411 */
	case UDM_HTTP_STATUS_PRECONDITION_FAILED:		/* 412 */
	case UDM_HTTP_STATUS_REQUEST_ENTITY_TOO_LARGE:		/* 413 */
	case UDM_HTTP_STATUS_REQUEST_URI_TOO_LONG:		/* 414 */	
	case UDM_HTTP_STATUS_UNSUPPORTED_MEDIA_TYPE:		/* 415 */
	case UDM_HTTP_STATUS_NOT_IMPLEMENTED:			/* 501 */
	case UDM_HTTP_STATUS_BAD_GATEWAY:			/* 502 */
	case UDM_HTTP_STATUS_NOT_SUPPORTED:			/* 505 */
	
		/*
		  FIXME: remove words from database
		  Check last reffering time to remove when
		  there are no links to this document anymore
		  UdmLog(Indexer,UDM_LOG_EXTRA,"Deleting URL");
		*/
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
	
	case UDM_HTTP_STATUS_INTERNAL_SERVER_ERROR:		/* 500 */
	case UDM_HTTP_STATUS_SERVICE_UNAVAILABLE:		/* 503 */
	case UDM_HTTP_STATUS_GATEWAY_TIMEOUT:			/* 504 */
	
		/* Keep words in database                */
		/* We'll retry later, maybe host is down */
		if((Host=UdmHostFind(&Indexer->Conf->Hosts,Doc->CurURL.hostname))){
			Host->net_errors++;
			Host=NULL;
		}
		next_index_time=time(NULL)+Doc->Spider.net_error_delay_time;
		UdmTime_t2HttpStr(next_index_time,dbuf);
		UdmVarListReplaceStr(&Doc->Sections,"Next-Index-Time",dbuf);
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
	
	default: /* Unknown status, retry later */
		UdmLog(Indexer,UDM_LOG_WARN,"HTTP %d We don't yet know how to handle it, skipped",status);
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
	}
	
	
	if(Doc->method==UDM_METHOD_GET && Doc->Spider.use_clones){
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_FINDORIG,db);
		if(result!=UDM_OK)return result;
		origin_id=UdmVarListFindInt(&Doc->Sections,"Origin-ID",0);
	}
	
	
	/* Check clones */
	if((origin_id)&&(origin_id!=url_id)){
		UdmLog(Indexer,UDM_LOG_EXTRA,"Duplicate Document with #%d",origin_id);
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_DELWORDS,db);
		if(use_crosswords){
			if(result==UDM_OK)result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_DELCWORDS,db);
		}
		if(result==UDM_OK)result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_UPDCLONE,db);
		return result;
	}
	
	/* Check that document wasn't modified since last indexing */
	if((UdmVarListFindInt(&Doc->Sections,"crc32old",0)==UdmVarListFindInt(&Doc->Sections,"crc32",0))&&
	   (!(Indexer->flags&UDM_FLAG_REINDEX))){
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_SUPDATE,db);
		return result;
	}
	
	/* Compose site_id string and calc it's CRC32 */
	sprintf(site_id_str,"%s://%s/",Doc->CurURL.schema,Doc->CurURL.hostinfo);
	UdmVarListAddInt(&Doc->Sections,"Site-ID",UdmStrCRC32(site_id_str));
	
	/* Copy default languages, if not given by server and not guessed */
	if(!(c=UdmVarListFindStr(&Doc->Sections,"Content-Language",NULL))){
		if((c=UdmVarListFindStr(&Doc->Sections,"Default-Content-Language",NULL)))
			UdmVarListAddStr(&Doc->Sections,"Content-Language",c);
	}
	
	/* For NEWS extension: get rec_id from my */
	/* parent out of db (if I have one...)    */
	if(use_newsext){
		UDM_VAR		*Sec;
		const char	*parent=NULL;
		int		parent_id=0;
		
		if((Sec=UdmVarListFind(&Doc->Sections,"References")) && Sec->val){
			/* References contains all message IDs of my */
			/* predecessors, space separated             */
			/* my direct parent is the last in the list  */
			if((parent = strrchr(Sec->val,' '))){
				/* parent now points to the space */
				/* character skip it              */
				++parent;
			}else{
				/* there is only one entry in */
				/* references, so this is my parent */
				parent=Sec->val;
			}	
		}
		
		/* get parent from database */
		if(parent && strlen(parent) && strchr(parent,'@')){
			char *pend, *pbeg, *psav;
			UDM_DOCUMENT Msg;
			
			/* Parent now points to the string in */
			/* format    <blabla@blabla.bla>      */
			/* Lets remove < > characters         */
			
			psav=pbeg=strdup(parent);
			if((pend=strrchr(parent,'>')))*pend=0;
			if(pbeg[0]=='<')pbeg++;
			
			UdmDocInit(&Msg);
			UdmVarListAddStr(&Msg.Sections,"Message-ID",parent);
			result = UdmURLAction(Indexer,&Msg,UDM_URL_ACTION_FINDBYMSG,db);
			parent_id = UdmVarListFindInt(&Msg.Sections,"ID",0);
			UdmVarListAddInt(&Doc->Sections,"Parent-ID",parent_id);
			free(psav);
			UdmDocFree(&Msg);
		}
		
		/* Now register me with my parent  */
		if(parent_id)result = UdmURLAction(Indexer,Doc,UDM_URL_ACTION_REGCHILD,db);
		
		if(result!=UDM_OK)return result;
	}
	
	/* Now store words and crosswords */
	if((Doc->method!=UDM_METHOD_CHECKMP3ONLY)||(Doc->is_mp3)){
		result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_INSWORDS,db);
		if(result!=UDM_OK)return result;
		if(use_crosswords){
			result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_INSCWORDS,db);
		}
		if(result!=UDM_OK)return result;
	}
	
	result=UdmURLAction(Indexer,Doc,UDM_URL_ACTION_LUPDATE,db);
	
	return result;
}


static int UdmDocUpdate(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc,void *vdb){
	size_t		maxsize=UdmVarListFindInt(&Indexer->Conf->Vars,"DocMemCacheSize",0);
	size_t		sec;
	int		flush=0;
	int		rc=UDM_OK;
	UDM_RESULT	*I=&Indexer->Conf->Indexed;
	UDM_DB		*db=vdb;
	
	if(Doc){
		/* Add document into cache */
		I->Doc=(UDM_DOCUMENT*)realloc(I->Doc,(I->num_rows+1)*sizeof(UDM_DOCUMENT));
		I->Doc[I->num_rows]=Doc[0];
		I->num_rows++;
		
		I->memused+=sizeof(UDM_DOCUMENT);
		/* Aproximation for Words memory usage  */
		I->memused+=Doc->Words.nwords*(sizeof(UDM_WORD)+10);
		/* Aproximation for CrossWords memory usage */
		I->memused+=Doc->CrossWords.ncrosswords*(sizeof(UDM_CROSSWORD)+10+20);
		/* Aproximation for Sections memory usage */
		for(sec=0;sec<Doc->Sections.nvars;sec++){
			I->memused+=sizeof(UDM_VAR);
			I->memused+=Doc->Sections.Var[sec].maxlen*9;
		}
	}
	
	UdmLog(Indexer,UDM_LOG_EXTRA,"DocCacheSize: %d/%d",I->memused,maxsize);
	if(I->memused>=maxsize)flush=1;
	if(I->num_rows>=1024)flush=1;
	if(!Doc)flush=1;
	
	if(flush){
		size_t	docnum;
		
		if (Indexer->Conf->Indexed.num_rows>1)
			UdmLog(Indexer, UDM_LOG_EXTRA, "Flush %d document(s)", Indexer->Conf->Indexed.num_rows);
		
		for (docnum=0;docnum<I->num_rows;docnum++){
			/* Flush all hrefs from cache in-memory    */
			/* cache into database. Note, this must    */
			/* be done before call of  StoreCrossWords */
			/* because we need to know all new URL IDs */
			
			UDM_THREADINFO(Indexer,"Updating",UdmVarListFindStr(&I->Doc[docnum].Sections,"URL",""));
			if(UDM_OK!=(rc=DocUpdate(Indexer,&I->Doc[docnum],db)))
				return rc;
		}
		if(UDM_OK!=(rc=UdmResAction(Indexer,&Indexer->Conf->Indexed,UDM_RES_ACTION_INSWORDS,db)))
			return rc;
		UdmResultFree(&Indexer->Conf->Indexed);
	}
	return rc;
}

__INDLIB__ int UdmURLAction(UDM_AGENT *A, UDM_DOCUMENT *D, int cmd,void *vdb){
	int res=UDM_ERROR;
	UDM_DB	*db=vdb;
	
	if(cmd==UDM_URL_ACTION_FLUSH)
		return UdmDocUpdate(A,D,vdb);
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmURLActionFiles(A,D,cmd,db);
#endif
#ifdef HAVE_SQL
	if(db->DBDriver!=UDM_DB_FILES)
		res=UdmURLActionSQL(A,D,cmd,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}


__INDLIB__ int UdmResAction(UDM_AGENT *A, UDM_RESULT *R, int cmd,void *vdb){
	int	res=UDM_ERROR;
	UDM_DB	*db=vdb;
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmResActionFiles(A,R,cmd,db);
#endif
#ifdef HAVE_SQL
	if(db->DBDriver!=UDM_DB_FILES)
		res=UdmResActionSQL(A,R,cmd,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}

__INDLIB__ int UdmCatAction(UDM_AGENT *A, UDM_CATEGORY *C, int cmd,void *vdb){
	UDM_DB	*db=vdb;
	int	res=UDM_ERROR;
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmCatActionFiles(A,C,cmd,db);
#endif
#ifdef HAVE_SQL
	if(db->DBDriver!=UDM_DB_FILES)
		res=UdmCatActionSQL(A,C,cmd,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}

__INDLIB__ int UdmSrvAction(UDM_AGENT *A, UDM_SERVERLIST *S, int cmd,void *vdb){
	UDM_DB	*db=vdb;
	int	res=UDM_ERROR;
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmSrvActionFiles(A,S,cmd,db);
#endif
#ifdef HAVE_SQL
	if(db->DBDriver!=UDM_DB_FILES)
		res=UdmSrvActionSQL(A,S,cmd,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}

__INDLIB__ UDM_RESULT *UdmFind(UDM_AGENT *A,const char * text){
	UDM_DB		*db=A->Conf->db;
	UDM_RESULT	*Res;
	int		res;
	unsigned long	ticks=UdmStartTimer();
	
	UdmLog(A,UDM_LOG_DEBUG,"Start UdmFind");
	
	/* Allocate result */
	Res=UdmResultInit(NULL);
	
	switch(db->DBDriver){
		case UDM_DB_SEARCHD:
			res=UdmSearchdFind(A,Res);
			break;
#ifdef HAVE_FILES
		case UDM_DB_FILES:
			res=UdmFindFiles(A,Res);
			break;
#endif
#ifdef HAVE_SQL
		default:
			res=UdmFindSQL(A,Res);
			break;
#endif
	}
	Res->work_time=ticks=UdmStartTimer()-ticks;
	UdmLog(A,UDM_LOG_DEBUG,"Done  UdmFind %.2f",(float)ticks/1000);
	return Res;
}


static int UdmStr2DBMode(const char * str1){
	int m = UDM_DBMODE_SINGLE;
	if(!UDM_STRNCASECMP(str1,"multi-crc"))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"crc-multi"))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"single"))m=UDM_DBMODE_SINGLE;
	else
	if(!UDM_STRNCASECMP(str1,"crc"))m=UDM_DBMODE_SINGLE_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"multi"))m=UDM_DBMODE_MULTI;
	else
	if(!UDM_STRNCASECMP(str1,"word2url"))m=UDM_DBMODE_WORD2URL;
	else
	if(!UDM_STRNCASECMP(str1,"cache"))m=UDM_DBMODE_CACHE;
	else
	if(!UDM_STRNCASECMP(str1,"blob"))m=UDM_DBMODE_BLOB;
	return(m);
}

int UdmDBSetAddr(void *vdb,const char *dbaddr, int mode){
	UDM_DB	*db=vdb;
	char	*s;
	
	if(UdmURLParse(&db->addr,dbaddr))return UDM_ERROR;
	db->open_mode=mode;
	db->DBADDR=strdup(dbaddr);
	
	if(!strcasecmp(db->addr.schema,"file")){
		db->DBType=UDM_DB_FILES;
	}else
	if(!strcasecmp(db->addr.schema,"searchd")){
		db->DBType=UDM_DB_SEARCHD;
	}else
	if(!strcasecmp(db->addr.schema,"msql")){
		db->DBType=UDM_DB_MSQL;
		db->DBSQL_LIMIT=1;
	}else
	if(!strcasecmp(db->addr.schema,"solid")){
		db->DBType=UDM_DB_SOLID;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}else
	if(!strcasecmp(db->addr.schema,"oracle7")){
		db->DBType=UDM_DB_ORACLE7;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
	}else
	if(!strcasecmp(db->addr.schema,"oracle8")){
		db->DBType=UDM_DB_ORACLE8;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
	}else
	if(!strcasecmp(db->addr.schema,"oracle")){
		db->DBType=UDM_DB_ORACLE8;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
	}else
	if(!strcasecmp(db->addr.schema,"mssql")){
		db->DBType=UDM_DB_MSSQL;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_TRUNCATE=1;
	}else
	if(!strcasecmp(db->addr.schema,"mysql")){
		db->DBType=UDM_DB_MYSQL;
		db->DBSQL_IN=1;
		db->DBSQL_LIMIT=1;
		db->DBSQL_GROUP=1;
	}else
	if(!strcasecmp(db->addr.schema,"pgsql")){
		db->DBType=UDM_DB_PGSQL;
		db->DBSQL_IN=1;
		db->DBSQL_LIMIT=1;
		db->DBSQL_GROUP=1;
		db->DBSQL_SELECT_FROM_DELETE=0;
	}else
	if(!strcasecmp(db->addr.schema,"ibase")){
		db->DBType=UDM_DB_IBASE;
		
		/* 
		while indexing large sites and using the SQL in statement 
		interbase will fail when the items in the in IN statements
		are more then 1500. We'd better have to fix code to avoid 
		big INs instead of hidding DBSQL_IN.
		*/
		
		/* db->DBSQL_IN=1; */ 
		db->DBSQL_GROUP=1;
	}else
	if(!strcasecmp(db->addr.schema,"sapdb")){
		db->DBType=UDM_DB_SAPDB;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}else
	if(!strcasecmp(db->addr.schema,"db2")){
		db->DBType=UDM_DB_DB2;
		db->DBSQL_IN=1;
		db->DBSQL_GROUP=1;
	}
	db->DBDriver=db->DBType;
	
	if((s=strchr(db->addr.filename,'?'))){
		char * tok, *lt;
		
		*s++='\0';
		tok=strtok_r(s,"&",&lt);
		while(tok){
			char * val;
			
			if((val=strchr(tok,'='))){
				*val++='\0';
				if(!strcmp(tok,"socket")&&val[0]){
					UDM_FREE(db->DBSock);
					db->DBSock=strdup(val);
				}
				if(!strcmp(tok,"numtables")&&val[0]){
					db->numtables=atoi(val);
					if(!db->numtables)
						db->numtables=1;
				}
				if(!strcmp(tok,"dbmode")&&val[0]){
					db->DBMode=UdmStr2DBMode(val);
				}
			}
			tok=strtok_r(NULL,"&",&lt);
		}
	}
	
	if(db->DBType==UDM_DB_IBASE){
		/* Ibase is a special case        */
		/* It's database name consists of */
		/* full path and file name        */ 
		db->DBName=strdup(db->addr.path);
		db->DBName[strlen(db->DBName)-1]='\0';
	}else{
		db->DBName=strdup(db->addr.path);
		sscanf(db->addr.path,"/%[^/]s",db->DBName);
	}
	if((s=strchr(db->addr.auth,':'))){
		*s=0;
		db->DBUser=strdup(db->addr.auth);
		db->DBPass=strdup(s+1);
		*s=':';
	}else{
		db->DBUser=strdup(db->addr.auth);
	}
	return UDM_OK;
}


__INDLIB__ int UdmStatAction(UDM_AGENT *A, UDM_STATLIST *S){
	UDM_DB	*db=A->Conf->db;
	int	res=UDM_ERROR;
	
#ifdef HAVE_FILES
	if(db->DBDriver==UDM_DB_FILES)
		res=UdmStatActionFiles(A,S,db);
#endif
#ifdef HAVE_SQL
	if(db->DBDriver!=UDM_DB_FILES)
		res=UdmStatActionSQL(A,S,db);
#endif
	if(res!=UDM_OK){
		strcpy(A->Conf->errstr,db->errstr);
		A->Conf->errcode=db->errcode;
	}
	return res;
}


/********************************************************/


UDM_DBLIST * UdmDBListInit(UDM_DBLIST * List){
	bzero(List,sizeof(*List));
	return(List);
}

size_t UdmDBListAdd(UDM_DBLIST * List,const char * addr){
	UDM_DB	*db;
	db=List->db=(UDM_DB*)realloc(List->db,(List->nitems+1)*sizeof(UDM_DB));
	db+=List->nitems;
	UdmDBInit(db);
	db->DBADDR=strdup(addr);
	db->searchd=0;
	List->nitems++;
	return(List->nitems);
}

void UdmDBListFree(UDM_DBLIST *List){
	size_t	i;
	UDM_DB	*db=List->db;
	
	for(i=0;i<List->nitems;i++){
		UdmDBFree(&db[i]);
	}
	UDM_FREE(List->db);
	UdmDBListInit(List);
}
